/****************************************************************************
gjutil.c:  Various utility routines - error checking malloc and
free, string functions etc...

   Copyright:  University of Oxford (1992, 1993)

TERMS OF USE:

The computer software and associated documentation called ALSCRIPT hereinafter
referred to as the WORK is more particularly identified and described in 
Appendix A.

The WORK was written and developed by: Geoffrey J. Barton

Laboratory of Molecular Biophysics
University of Oxford
Rex Richards Building
South Parks Road
Oxford OX1 3QU
U.K.

Tel:  (+44) 865-275368
Fax:  (+44) 865-510454

Internet: gjb@bioch.ox.ac.uk
Janet:    gjb@uk.ac.ox.bioch

The WORK is Copyright (1992,1993) University of Oxford

Administrative Offices
Wellington Square
Oxford OX1 2JD
U.K.

CONDITIONS:

The WORK is made available for educational and non-commercial research 
purposes.

For commercial use, a commercial licence is required - contact the author
at the above address for details.

The WORK may be copied and redistributed, provided that no charge is
made for the WORK, that the WORK is not incorporated into a program or
package for sale and that this text, all copyright notices and the
authors' address is left unchanged on every copy.

The WORK may be modified, however this text, all copyright notices and
the authors' address must be left unchanged on every copy, and all
changes must be documented after this notice.  A copy of the
modified WORK must be supplied to the author.

******************************************************************************/
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "gjutil.h"

/* define pointers for standard streams to allow redefinition to files */

FILE *std_err;
FILE *std_in;
FILE *std_out;

void *GJmalloc(size_t size)
/* malloc with simple error check */
/* G. J. Barton 1992 */
{
	void *ptr;
	ptr = (void *) malloc(size);
	if(ptr == NULL){
		GJerror("malloc error");
		exit(0);
	}
	return ptr;
}

void *GJrealloc(void *ptr,size_t size)
/* realloc with error check */
/* G. J. Barton 1992 */
{
	ptr = (void *) realloc(ptr,size);
	if(ptr == NULL){
		GJerror("realloc error");
		exit(0);
	}
	return ptr;
}
void *GJmallocNQ(size_t size)
/* as for GJmalloc, but returns NULL on error*/
/* G. J. Barton 1992 */
{
	void *ptr;
	ptr = (void *) malloc(size);
	if(ptr == NULL){
		GJerror("malloc error");
                return NULL;
	}
	return ptr;
}

void *GJreallocNQ(void *ptr,size_t size)
/* as for GJrealloc with error check but returns NULL on error*/
/* G. J. Barton 1992 */
{
	ptr = (void *) realloc(ptr,size);
	if(ptr == NULL){
		GJerror("realloc error");
		return NULL;
	}
	return ptr;
}
void GJfree(void *ptr)
/* free with error check */
/* G. J. Barton 1992 */
{
	if(ptr == NULL){
		GJerror("Attempt to free NULL pointer");
		exit(0);
	}
	free(ptr);
}

void GJerror(const char *prefix)
/* writes error message contained in prefix and contents of errno
   to std_err.
*/
/* G. J. Barton 1992 */
{
	if(prefix != NULL){
		if(*prefix != '\0'){
			fprintf(std_err,"%s: ",prefix);
		}
	}
	fprintf(std_err,"%s\n",strerror(errno));
}

/*
error:   calls GJerror 
*/
void error(const char *str,int flag)
{
    GJerror(str);
    if(flag)exit(0);
}


char *GJstoupper(const char *s)
/* return a copy of s in upper case */
/* G. J. Barton 1992 */
{
	char *temp;
	int i;
	temp = GJstrdup(s);
	i=0;
	while(temp[i] != '\0'){
		temp[i] = toupper(temp[i]);
		++i;
	}
	return temp;
}
char *GJstolower(const char *s)
/* return a copy of s in lower case */
/* G. J. Barton 1992 */
{
	char *temp;
	int i;
	temp = GJstrdup(s);
	i=0;
	while(temp[i] != '\0'){
		temp[i] = tolower(temp[i]);
		++i;
	}
	return temp;
}
char *GJstoup(char *s)
/* return s in upper case */
/* G. J. Barton 1992 */
{
	int i;
	i=0;
	while(s[i] != '\0'){
		s[i] = toupper(s[i]);
		++i;
	}
	return s;
}
char *GJstolo(char *s)
/* return s in lower case */
/* G. J. Barton 1992 */
{
	int i;
	i=0;
	while(s[i] != '\0'){
		s[i] = tolower(s[i]);
		++i;
	}
	return s;
}  

char *GJstrdup(const char *s)
/* returns a pointer to a copy of string s */
/* G. J. Barton 1992 */

{
	char *temp;
	temp = (char *) GJmalloc(sizeof(char) * (strlen(s)+1));
	temp = strcpy(temp,s);
	return temp;
}

FILE *GJfopen(const char *fname,const char *type,int action)
/* a file open function with error checking.  The third argument
is set to 0 if we want a failed open to return, or 1 if we
want a failed open to exit the program.
*/
/* G. J. Barton 1992 */
{
	FILE *ret_val;
	ret_val = fopen(fname,type);
	if(ret_val == NULL){
		GJerror(strcat("Cannot Open File: ",fname));
		if(action == 1){
			exit(1);
		}
	}
	return ret_val;
}

int GJfclose(FILE *fp,int action)
/* a file close function with error checking.  The second argument
is set to 0 if we want a failed close to return, or 1 if we
want a failed close to exit the program.
*/
/* G. J. Barton 1992 */
{
	int ret_val;
	ret_val = fclose(fp);
	if(ret_val != 0){
		GJerror("Error closing File");
		if(action == 1){
			exit(1);
		}
	}
	return ret_val;
}

void GJinitfile()
/* just set the standard streams */
{
	std_err = stderr;
	std_in = stdin;
	std_out = stdout;
}

char *GJfnonnull(char *string)
/* return pointer to first non null character in the string */
/* this could cause problems if the string is not null terminated */
{
	while(*string != '\0'){
		++string;
	}
	return ++string;
}

char *GJstrappend(char *string1, char *string2)
/* appends string2 to the end of string2.  Any newline characters are removed
from string1, then the first character of string2 overwrites the null at the
end of string1.
string1 and string2 must have been allocated with malloc.
*/
/* G. J. Barton July 1992 */
{
	char *ret_val;
	ret_val = GJremovechar(string1,'\n');
	ret_val = (char *) GJrealloc(ret_val,
			   sizeof(char) * (strlen(ret_val) + strlen(string2) + 1));
        ret_val = strcat(ret_val,string2);
        return ret_val;
}

char *GJremovechar(char *string,char c)
/* removes all instances of character c from string
   returns a pointer to the reduced, null terminated string
*/
/* G. J. Barton (July 1992) */
{
	char *temp;
	int j,i,nchar;
	nchar = 0;
	i=0;
	while(string[i] != '\0'){
		if(string[i] == c){
			++nchar;
		}
		++i;
	}
	if(nchar == 0){
		 return string;
	}else{
		temp = (char *) GJmalloc(sizeof(char) * (strlen(string)-nchar));
		j=0;
		i=0;
		while(string[i] != '\0'){
			if(string[i] != c){
				temp[j] = string[i];
				++j;
			}
			++i;
		}
		temp[++j] = '\0';
		GJfree(string);
		return temp;
	}
}

char *GJsubchar(char *string,char c2,char c1)
/* substitutes c1 for c2 in string
*/
/* G. J. Barton (July 1992) */
{
	int j,i,nchar;
	nchar = 0;
	i=0;
	while(string[i] != '\0'){
		if(string[i] == c1){
                    string[i] = c2;
		}
		++i;
	}
	return string;
}

/* create a string and if fchar != NULL fill with characters  */
/* always set the len-1 character to '\0' */

char *GJstrcreate(size_t len,char *fchar)
{
	char *ret_val;
	ret_val = (char *) GJmalloc(sizeof(char) * len);
	--len;
	ret_val[len] = '\0';
	if(fchar != NULL){
		while(len > -1){
			ret_val[len] = *fchar;
			--len;
		}
	}
	return ret_val;
}

/* searches for string s2 in string s1 and returns pointer to first instance
of s2 in s1 or NULL if no instance found.  s1 and s2 must be null terminated
*/		
char *GJstrlocate(char *s1, char *s2)
{
    int i=0;
    int j=0;
    int k;
    if(strlen(s1) == 0 || strlen(s2) == 0) return NULL;
    while(i<strlen(s1)){
        j=0;
        k=i;
        while(j<strlen(s2) && s1[k] == s2[j]){
                ++k;
                ++j;
        }
        if(j == strlen(s2)) return &s1[i];
        ++i;
    }
    return NULL;
}
#include <stdlib.h>
#include <string.h>


/* GJstrtok()

This version of strtok places the work pointer at the location of the first 
character in the next token, rather than just after the last character of the 
current token.  This is useful for extracting quoted strings 
*/

char *GJstrtok(char *input_string,const char *token_list)
{
  static char *work;
  char *return_ptr;

  if(input_string != NULL){
    /* first call */
    work =  input_string;
  }

  /* search for next non-token character */
  while(strchr(token_list,*work)!=NULL){
    ++work;
  }

  if(*work == '\0'){
    /* if we've reached the end of string, then return NULL */
    return NULL;
  }else{
    return_ptr = (char *) work;
    while(strchr(token_list,*work) == NULL){
      if(*work == '\0'){
	/* end of the string */
	return return_ptr;
      }else{
	++work;
      }
    }
    *work = '\0';
    ++work;
    /* now increment work until we find the next non-delimiter character */
    while(strchr(token_list,*work) != NULL){
      if(*work == '\0'){
	break;
      }else{
	++work;
      }
    }
    return return_ptr;
  }
}
/**************************************************************************
return a pointer to space for a rectangular unsigned character array
Version 2.0  ANSI and uses GJmallocNQ
--------------------------------------------------------------------------*/

unsigned char **uchararr(int i,int j)
{
    unsigned char **temp;
    int k, rowsiz;

    temp = (unsigned char **) GJmallocNQ(sizeof(unsigned char *) * i);
    if(temp == NULL) return NULL;

    rowsiz = sizeof(unsigned char) * j;

    for (k = 0; k < i; ++k){
	temp[k] =  (unsigned char *) GJmallocNQ(rowsiz);
	if(temp[k] == NULL) return NULL;
    }
    return temp;
}

/**************************************************************************
return a pointer to space for a rectangular signed character array
Version 2.0  ANSI
--------------------------------------------------------------------------*/
#ifdef VAX
char **chararr(int i,int j)
#else
signed char **chararr(int i,int j)
#endif

{
    signed char **temp;
    int k, rowsiz;

#ifdef VAX
    temp = (char **) GJmallocNQ(sizeof(char *) * i);
#else
    temp = (signed char **) GJmallocNQ(sizeof(char *) * i);
#endif
    if(temp == NULL) return NULL;

    rowsiz = sizeof(char) * j;

    for (k = 0; k < i; ++k){
	temp[k] =  (signed char *) GJmallocNQ(rowsiz);
	if(temp[k] == NULL) return NULL;
    }
    return temp;
}


/* mcheck - check a call to malloc - if the call has failed, print the
error message and exit the program */
/* ANSI Version - also uses GJerror routine and ptr is declared void*/

void mcheck(void *ptr,char *msg)

{
    if(ptr == NULL){
        GJerror("malloc/realloc error");
	exit(0);
    }
}

/* set a string to blanks and add terminating nul */
char *GJstrblank(char *string,int len)

{
  --len;
  string[len] = '\0';
  --len;
  while(len > -1){
    string[len] = ' ';
    --len;
  }
  return string;
}

/* Initialise an unsigned char array */  
void GJUCinit(unsigned char **array,int i,int j,unsigned char val)
{
  int k,l;

  for(k=0;k<i;++k){
    for(l=0;l<j;++l){
      array[k][l] = val;
    }
  }
}
/*Initialise a signed char array */

#ifdef VAX
void GJCinit(char **array,int i,int j,char val)
#else
void GJCinit(signed char **array,int i,int j,char val)
#endif

{
  int k,l;

  for(k=0;k<i;++k){
    for(l=0;l<j;++l){
      array[k][l] = val;
    }
  }
}

/******************************************************************
GJcat:  concatenate N NULL terminated strings into a single string.
The source strings are not altered
Author: G. J. Barton (Feb 1993)
------------------------------------------------------------------*/
char *GJcat(int N,...)
{
	va_list parminfo;
	int i,j,k;
	char **values;	/*input strings */
	int *value_len; /*lengths of input strings */
	int ret_len;    /*length of returned string */
	char *ret_val;  /*returned string */

	ret_len = 0;
	values = (char **) GJmalloc(sizeof(char *) * N);
	value_len = (int *) GJmalloc(sizeof(int *) * N);

	va_start(parminfo,N);

	/* Get pointers and lengths for the N arguments */
	for(i=0;i<N;++i){
		values[i] = va_arg(parminfo,char *);
		value_len[i] = strlen(values[i]);
		ret_len += value_len[i];
	}
	
	ret_val = (char *) GJmalloc(sizeof(char) * (ret_len+1));

	/* Transfer input strings to output string */
	k=0;
	for(i=0;i<N;++i){
		for(j=0;j<value_len[i];++j){
			ret_val[k] = values[i][j];
			++k;
		}
	}
	ret_val[k] = '\0';
	GJfree(values);
	GJfree(value_len);

	va_end(parminfo);

	return ret_val;
}

/************************************************************************

GJGetToken:

The aim of this routine is to emulate the strtok() function, but reading
from a file.  The functionality may differ slightly...

Characters are read from the file until a character that is found in
delim is encountered.  When this occurs, token is returned.  If the
file consists entirely of delimiters, then token is freed
and NULL is returned.  Similarly, if end of file is encountered.

------------------------------------------------------------------------*/


char *GJGetToken(FILE *in, const char *delim)

{
	int i;
	char c;
	char *token;

	i=0;

	token = GJmalloc(sizeof(char));
	
	while((c=fgetc(in)) != EOF){
		if(strchr(delim,c) == NULL){
			/* not a delimiter */
			token[i++] = c;
			token = GJrealloc(token,sizeof(char) * (i+1));
		}else if(i>0){
		        token[i] = '\0';
			return token;
		}
	}
/*	GJerror("End of File Encountered");*/
	GJfree(token);
	return NULL;
} 
